/*
 * Samsung Exynos5 SoC series Sensor driver
 *
 *
 * Copyright (c) 2011 Samsung Electronics Co., Ltd
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/version.h>
#include <linux/gpio.h>
#include <linux/clk.h>
#include <linux/regulator/consumer.h>
#include <linux/videodev2.h>
#include <linux/videodev2_exynos_camera_ext.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#ifdef CONFIG_OF
#include <linux/of_gpio.h>
#endif
#include <mach/regs-gpio.h>
#include <mach/regs-clock.h>
#include <plat/clock.h>
#include <plat/gpio-cfg.h>
#include <media/v4l2-ctrls.h>
#include <media/v4l2-device.h>
#include <media/v4l2-subdev.h>
#include <mach/exynos-fimc-is-sensor.h>

#include "../fimc-is-core.h"
#include "../fimc-is-device-sensor.h"
#include "../fimc-is-resourcemgr.h"
#include "../fimc-is-hw.h"
#include "fimc-is-device-ov5640.h"

#define SENSOR_NAME "OV5640"
#define DEFAULT_SENSOR_WIDTH	184
#define DEFAULT_SENSOR_HEIGHT	104
#define SENSOR_MEMSIZE DEFAULT_SENSOR_WIDTH * DEFAULT_SENSOR_HEIGHT

#define REG_TERM 0xfffe
#define VAL_TERM 0xfe
#define REG_DLY  0xffff
#define OV5640_XCLK_MIN 6000000
#define OV5640_XCLK_MAX 24000000

int sensor_ov5640_stream_off(struct v4l2_subdev *subdev);
int sensor_ov5640_stream_on(struct v4l2_subdev *subdev);

static struct fimc_is_sensor_cfg config_ov5640[] = {
	/* 320x180@10fps : only for vision(settle) */
	FIMC_IS_SENSOR_CFG( 640,  480, 30, 12, 0),
	FIMC_IS_SENSOR_CFG(1280,  720, 30, 12, 1),
	FIMC_IS_SENSOR_CFG(1920, 1080, 30, 12, 2),
	FIMC_IS_SENSOR_CFG(2560, 1920, 30, 12, 3),
};

static struct fimc_is_vci vci_ov5640[] = {
	{
		.pixelformat = V4L2_PIX_FMT_YUYV,
		.vc_map = {2, 1, 0, 3}
	}, {
		.pixelformat = V4L2_PIX_FMT_JPEG,
		.vc_map = {0, 2, 1, 3}
	}
};

static int ov5640_write(struct v4l2_subdev *subdev, uint16_t reg, uint8_t val)
{
	struct fimc_is_module_enum *module;
	struct fimc_is_module_ov5640 *module_ov5640;
	struct i2c_client *client;

	BUG_ON(!subdev);
	module = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	module_ov5640 = module->private_data;
	client = module->client;

	return fimc_is_sensor_write(client, reg, val);
}

/**
* @short I2C Read operation
* @param[in] i2c_client I2C client
* @param[in] reg register address
* @param[out] val value read
* @return Error code
*/
static int ov5640_read(struct v4l2_subdev *subdev, uint16_t reg, uint8_t *val)
{
	int ret = 0;
	struct fimc_is_module_enum *module;
	struct fimc_is_module_ov5640 *module_ov5640;
	struct i2c_client *client;
	uint8_t rval;

	BUG_ON(!subdev);
	module = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	module_ov5640 = module->private_data;
	client = module->client;

	ret = fimc_is_sensor_read(client, reg, &rval);
	*val = rval;

	return ret;
}

static int ov5640_write_array(struct v4l2_subdev *subdev,
				struct regval_list *regs, int array_size)
{
	int ret = 0;
	struct fimc_is_module_enum *module;
	struct fimc_is_module_ov5640 *module_ov5640;
	struct i2c_client *client;
	int i = 0;

	BUG_ON(!subdev);
	module = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	module_ov5640 = module->private_data;
	client = module->client;

	if (!regs)
		return -EINVAL;

	while (i < array_size) {
		if (regs->addr == REG_DLY)
			mdelay(regs->data);
		else
			ret = fimc_is_sensor_write(client, regs->addr, regs->data);
		if (ret == -EIO)
			return ret;
		i++;
		regs++;
	}
	return 0;
}

static int ov5640_burst_write(struct v4l2_subdev *subdev, u16 base_addr,
				u16 size, u8 *wr_buf)
{
	int ret = 0;
	struct fimc_is_module_enum *module;
	struct fimc_is_module_ov5640 *module_ov5640;
	struct i2c_client *client;
	int i = 0;

	BUG_ON(!subdev);
	module = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	module_ov5640 = module->private_data;
	client = module->client;

	while (i < size) {
		ret = fimc_is_sensor_write(client, base_addr+i, *wr_buf);
		if (ret < -EIO)
			return ret;
		i++;
		wr_buf++;
	}
	return 0;
}

static int ov5640_sleep(struct v4l2_subdev *subdev)
{
	int ret = 0;
	u8 rw_buf = 0;

printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	ret = ov5640_read(subdev, OV5640_SYSTEM_CTRL0, &rw_buf);
	if (ret < 0)
		return ret;

	rw_buf |= OV5640_SYSTEM_POWER_DOWN_MASK;
	ret = ov5640_write(subdev, OV5640_SYSTEM_CTRL0, rw_buf);

	return ret;
}

static int ov5640_wakeup(struct v4l2_subdev *subdev)
{
	int ret = 0;
	u8 rw_buf = 0x21;

printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	ret = ov5640_write(subdev, OV5640_POLARITY_CTRL00, rw_buf);
	if (ret < 0)
		return ret;

	rw_buf = 0;
	ret = ov5640_read(subdev, OV5640_SYSTEM_CTRL0, &rw_buf);
	if (ret < 0)
		return ret;

	rw_buf &= (~OV5640_SYSTEM_POWER_DOWN_MASK);
	ret = ov5640_write(subdev, OV5640_SYSTEM_CTRL0, rw_buf);
	if (ret < 0)
		return ret;
	return ret;
}

static int ov5640_soft_reset(struct v4l2_subdev *subdev)
{
	int ret = 0;
	u8 rw_buf = 0;

printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	ret = ov5640_read(subdev, OV5640_SYSTEM_CTRL0, &rw_buf);
	if (ret < 0)
		return ret;

	rw_buf |= OV5640_SYSTEM_RESET_MASK;
	ret = ov5640_write(subdev, OV5640_SYSTEM_CTRL0, rw_buf);

	return ret;
}

static int __sensor_init(struct v4l2_subdev *subdev)
{
	int ret;
	u8 count;
	u8 chip_id[2];
	
	ov5640_soft_reset(subdev);

	ov5640_read(subdev, 0x300a, &chip_id[0]);
	ov5640_read(subdev, 0x300b, &chip_id[1]);
printk("CKKIM -> %s[%d] : chip_id = %x%x\n",__func__,__LINE__, chip_id[0],chip_id[1]);

	ret = ov5640_write_array(subdev, OV5640_base_init_regs,
					ARRAY_SIZE(OV5640_base_init_regs));
	if (ret < 0) {
		v4l2_err(subdev, "write OV5640_base_init_regs error\n");
		return ret;
	}
	mdelay(100);
	ret = ov5640_sleep(subdev);
	mdelay(100);
	ret = ov5640_write_array(subdev, OV5640_VGA_regs,
					ARRAY_SIZE(OV5640_VGA_regs));
	if (ret < 0) {
		v4l2_err(subdev, "write sensor_vga_regs error\n");
		return ret;
	}
	mdelay(100);
	ret = ov5640_wakeup(subdev);
	if (ret < 0) {
		v4l2_err(subdev, "ov5640_wakeup error\n");
		return ret;
	}
	mdelay(100);

	ov5640_read(subdev, 0x4300, &chip_id[0]);
printk("CKKIM -> %s[%d] : 0x4300 = [0x%02x]\n",__func__,__LINE__, chip_id[0]);

//	ret = ov5640_write_array(subdev, ov5640_test_regs,
//					ARRAY_SIZE(ov5640_test_regs));
//	if (ret < 0) {
//		v4l2_err(subdev, "write ov5640_test_regs error\n");
//		return ret;
//	}

	return 0;
}

static int sensor_ov5640_open(struct v4l2_subdev *sd,
	struct v4l2_subdev_fh *fh)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);

	pr_info("%s\n", __func__);
	return 0;
}
static int sensor_ov5640_close(struct v4l2_subdev *sd,
	struct v4l2_subdev_fh *fh)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	pr_info("%s\n", __func__);
	return 0;
}
static int sensor_ov5640_registered(struct v4l2_subdev *sd)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	pr_info("%s\n", __func__);
	return 0;
}

static void sensor_ov5640_unregistered(struct v4l2_subdev *sd)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	pr_info("%s\n", __func__);
}

static const struct v4l2_subdev_internal_ops internal_ops = {
	.open = sensor_ov5640_open,
	.close = sensor_ov5640_close,
	.registered = sensor_ov5640_registered,
	.unregistered = sensor_ov5640_unregistered,
};

static int sensor_ov5640_init(struct v4l2_subdev *subdev, u32 val)
{
	int ret = 0;
	struct fimc_is_module_enum *module;

printk("CKKIM -> %s[%d] \n",__func__,__LINE__);

	BUG_ON(!subdev);
	module = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);

	ret = __sensor_init(subdev);
	if (ret < 0)
		return ret;
	pr_info("[MOD:D:%d] %s(%d)\n", module->id, __func__, val);

	return ret;
}

static int sensor_ov5640_s_ctrl(struct v4l2_subdev *subdev, struct v4l2_control *ctrl)
{
	int ret = 0;

	switch (ctrl->id) {
	case V4L2_CID_SCENEMODE:
	case V4L2_CID_FOCUS_MODE:
	case V4L2_CID_WHITE_BALANCE_PRESET:
	case V4L2_CID_IMAGE_EFFECT:
	case V4L2_CID_CAM_ISO:
	case V4L2_CID_CAM_CONTRAST:
	case V4L2_CID_CAM_SATURATION:
	case V4L2_CID_CAM_SHARPNESS:
	case V4L2_CID_CAM_BRIGHTNESS:
	case V4L2_CID_CAM_METERING:
	case V4L2_CID_CAM_SET_AUTO_FOCUS:
	case V4L2_CID_CAM_OBJECT_POSITION_X:
	case V4L2_CID_CAM_OBJECT_POSITION_Y:
	case V4L2_CID_CAM_FACE_DETECTION:
	case V4L2_CID_CAM_WDR:
	case V4L2_CID_CAM_AUTO_FOCUS_RESULT:
	case V4L2_CID_JPEG_QUALITY:
	case V4L2_CID_CAM_AEAWB_LOCK_UNLOCK:
	case V4L2_CID_CAM_CAF_START_STOP:
	case V4L2_CID_CAM_ZOOM:
		err("invalid ioctl(0x%08X) is requested", ctrl->id);
		break;
	case V4L2_CID_CAM_SINGLE_AUTO_FOCUS:
		break;
	case V4L2_CID_CAPTURE:
		break;
	case V4L2_CID_CAM_FLASH_MODE:
		break;
	case V4L2_CID_CAM_FRAME_RATE:
		break;
	default:
		err("unsupported ioctl(0x%08X) is requested", ctrl->id);
		//ret = -EINVAL;
		ret = 0;
		break;
	}

	return ret;
}

static const struct v4l2_subdev_core_ops core_ops = {
	.init = sensor_ov5640_init,
	.s_ctrl	= sensor_ov5640_s_ctrl,
};

static int sensor_ov5640_s_stream(struct v4l2_subdev *subdev, int enable)
{
	int ret = 0;
	struct fimc_is_module_enum *sensor;

	pr_info("%s\n", __func__);

	sensor = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	if (!sensor) {
		err("sensor is NULL");
		ret = -EINVAL;
		goto p_err;
	}

	if (enable) {
		ret = CALL_MOPS(sensor, stream_on, subdev);
		if (ret) {
			err("s_duration is fail(%d)", ret);
			goto p_err;
		}
	} else {
		ret = CALL_MOPS(sensor, stream_off, subdev);
		if (ret) {
			err("s_duration is fail(%d)", ret);
			goto p_err;
		}
	}

p_err:
	return 0;
}

static int sensor_ov5640_s_param(struct v4l2_subdev *subdev, struct v4l2_streamparm *param)
{
	int ret = 0;
	struct fimc_is_module_enum *sensor;
	struct v4l2_captureparm *cp;
	struct v4l2_fract *tpf;
	u64 duration;

	BUG_ON(!subdev);
	BUG_ON(!param);

printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	pr_info("%s\n", __func__);

	cp = &param->parm.capture;
	tpf = &cp->timeperframe;

	if (!tpf->denominator) {
		err("denominator is 0");
		ret = -EINVAL;
		goto p_err;
	}

	if (!tpf->numerator) {
		err("numerator is 0");
		ret = -EINVAL;
		goto p_err;
	}

	duration = (u64)(tpf->numerator * 1000 * 1000 * 1000) /
					(u64)(tpf->denominator);

	sensor = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	if (!sensor) {
		err("sensor is NULL");
		ret = -EINVAL;
		goto p_err;
	}

	ret = CALL_MOPS(sensor, s_duration, subdev, duration);
	if (ret) {
		err("s_duration is fail(%d)", ret);
		goto p_err;
	}

p_err:
	return ret;
}

static int sensor_ov5640_s_format(struct v4l2_subdev *subdev, struct v4l2_mbus_framefmt *fmt)
{
	int ret = 0;
	struct fimc_is_module_enum *module;
	struct fimc_is_module_ov5640 *module_ov5640;
	struct i2c_client *client;

	BUG_ON(!subdev);
	BUG_ON(!fmt);

	module = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	if (!module) {
		err("module is NULL");
		ret = -EINVAL;
		goto p_err;
	}

	module_ov5640 = module->private_data;
	if (unlikely(!module_ov5640)) {
		err("module_ov5640 is NULL");
		ret = -EINVAL;
		goto p_err;
	}

	client = module->client;
	if (unlikely(!client)) {
		err("client is NULL");
		ret = -EINVAL;
		goto p_err;
	}

	module_ov5640->image.window.offs_h = 0;
	module_ov5640->image.window.offs_v = 0;
	module_ov5640->image.window.width = fmt->width;
	module_ov5640->image.window.height = fmt->height;
	module_ov5640->image.window.o_width = fmt->width;
	module_ov5640->image.window.o_height = fmt->height;
	module_ov5640->image.format.pixelformat = fmt->code;

	info("[352] output size : %d x %d\n", fmt->width, fmt->height);

p_err:
	return ret;
}

static const struct v4l2_subdev_video_ops video_ops = {
	.s_stream = sensor_ov5640_s_stream,
	.s_parm = sensor_ov5640_s_param,
	.s_mbus_fmt = sensor_ov5640_s_format
};

static const struct v4l2_subdev_ops subdev_ops = {
	.core = &core_ops,
	.video = &video_ops
};

#define OV5640_PAD_OUTPUT_ENABLE_01	0x3017
#define OV5640_PAD_OUTPUT_ENABLE_02	0x3018
#define	OV5640_PAD_DISABLE_ALL		0x00
#define	OV5640_PAD_ENABLE_ALL		0xff

int sensor_ov5640_stream_on(struct v4l2_subdev *subdev)
{
	int ret = 0;
	struct fimc_is_module_enum *sensor;
	struct i2c_client *client;

printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	BUG_ON(!subdev);

	sensor = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	if (unlikely(!sensor)) {
		err("sensor is NULL");
		ret = -EINVAL;
		goto p_err;
	}

	client = sensor->client;
	if (unlikely(!client)) {
		err("client is NULL");
		ret = -EINVAL;
		goto p_err;
	}

//	ret = __sensor_init(subdev);
//	if (ret < 0)
//		return ret;

//	ret = ov5640_write_array(subdev, ov5640_preview_settings_tbl,
//					ARRAY_SIZE(ov5640_preview_settings_tbl));
//	if (ret < 0) {
//		v4l2_err(subdev, "write sensor_vga_regs error\n");
//		return ret;
//	}
//
//	ret = ov5640_wakeup(subdev);
//	if (ret < 0) {
//		v4l2_err(subdev, "ov5640_wakeup error\n");
//		return ret;
//	}
p_err:
	return 0;
}

int sensor_ov5640_stream_off(struct v4l2_subdev *subdev)
{
	int ret = 0;
	struct fimc_is_module_enum *sensor;
	struct i2c_client *client;
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);

	BUG_ON(!subdev);

	sensor = (struct fimc_is_module_enum *)v4l2_get_subdevdata(subdev);
	if (unlikely(!sensor)) {
		err("sensor is NULL");
		ret = -EINVAL;
		goto p_err;
	}

	client = sensor->client;
	if (unlikely(!client)) {
		err("client is NULL");
		ret = -EINVAL;
		goto p_err;
	}

	fimc_is_sensor_write(client, OV5640_PAD_OUTPUT_ENABLE_01, OV5640_PAD_DISABLE_ALL);
	fimc_is_sensor_write(client, OV5640_PAD_OUTPUT_ENABLE_01, OV5640_PAD_DISABLE_ALL);

p_err:
	return 0;
}

/*
 * @ brief
 * frame duration time
 * @ unit
 * nano second
 * @ remarks
 */
int sensor_ov5640_s_duration(struct v4l2_subdev *subdev, u64 duration)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return 0;
}

int sensor_ov5640_g_min_duration(struct v4l2_subdev *subdev)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return 0;
}

int sensor_ov5640_g_max_duration(struct v4l2_subdev *subdev)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return 0;
}

int sensor_ov5640_s_exposure(struct v4l2_subdev *subdev, u64 exposure)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return 0;
}

int sensor_ov5640_g_min_exposure(struct v4l2_subdev *subdev)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return 0;
}

int sensor_ov5640_g_max_exposure(struct v4l2_subdev *subdev)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return 0;
}

int sensor_ov5640_s_again(struct v4l2_subdev *subdev, u64 sensitivity)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return 0;
}

int sensor_ov5640_g_min_again(struct v4l2_subdev *subdev)
{
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return 0;
}

int sensor_ov5640_g_max_again(struct v4l2_subdev *subdev)
{
	int ret = 0;
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return ret;
}

int sensor_ov5640_s_dgain(struct v4l2_subdev *subdev)
{
	int ret = 0;
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return ret;
}

int sensor_ov5640_g_min_dgain(struct v4l2_subdev *subdev)
{
	int ret = 0;
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return ret;
}

int sensor_ov5640_g_max_dgain(struct v4l2_subdev *subdev)
{
	int ret = 0;
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	return ret;
}

struct fimc_is_sensor_ops module_ov5640_ops = {
	.stream_on	= sensor_ov5640_stream_on,
	.stream_off	= sensor_ov5640_stream_off,
	.s_duration	= sensor_ov5640_s_duration,
	.g_min_duration	= sensor_ov5640_g_min_duration,
	.g_max_duration	= sensor_ov5640_g_max_duration,
	.s_exposure	= sensor_ov5640_s_exposure,
	.g_min_exposure	= sensor_ov5640_g_min_exposure,
	.g_max_exposure	= sensor_ov5640_g_max_exposure,
	.s_again	= sensor_ov5640_s_again,
	.g_min_again	= sensor_ov5640_g_min_again,
	.g_max_again	= sensor_ov5640_g_max_again,
	.s_dgain	= sensor_ov5640_s_dgain,
	.g_min_dgain	= sensor_ov5640_g_min_dgain,
	.g_max_dgain	= sensor_ov5640_g_max_dgain
};

int sensor_ov5640_probe(struct i2c_client *client,
	const struct i2c_device_id *id)
{
	int ret = 0;
	struct fimc_is_core *core;
	struct v4l2_subdev *subdev_module;
	struct fimc_is_module_enum *module;
	struct fimc_is_device_sensor *device;
	struct sensor_open_extended *ext;
	static bool probe_retried = false;
	struct device *dev = &client->dev;
	struct fimc_is_module_ov5640 *module_ov5640;
	
printk("CKKIM -> %s[%d] \n",__func__,__LINE__);
	if (!fimc_is_dev)
		goto probe_defer;

	core = (struct fimc_is_core *)dev_get_drvdata(fimc_is_dev);
	if (!core) {
		err("core device is not yet probed");
		return -EPROBE_DEFER;
	}

	device = &core->sensor[SENSOR_OV5640_INSTANCE];

	subdev_module = kzalloc(sizeof(struct v4l2_subdev), GFP_KERNEL);
	if (!subdev_module) {
		err("subdev_module is NULL");
		ret = -ENOMEM;
		goto p_err;
	}

	/* OV5640 */
	module = &device->module_enum[atomic_read(&core->resourcemgr.rsccount_module)];
	atomic_inc(&core->resourcemgr.rsccount_module);
	module->id = SENSOR_OV5640_NAME;
	module->subdev = subdev_module;
	module->device = SENSOR_OV5640_INSTANCE;
	module->ops = &module_ov5640_ops;
	module->client = client;
	module->active_width = 640;
	module->active_height = 480;
	module->pixel_width = module->active_width + 16;
	module->pixel_height = module->active_height + 10;
	module->max_framerate = 30;
	module->position = SENSOR_POSITION_REAR;
	module->mode = CSI_MODE_VC_ONLY;
	module->lanes = CSI_DATA_LANES_2;
	module->vcis = ARRAY_SIZE(vci_ov5640);
	module->vci = vci_ov5640;
	module->setfile_name = "setfile_6b2.bin";
	module->cfgs = ARRAY_SIZE(config_ov5640);
	module->cfg = config_ov5640;
	module->private_data = kzalloc(sizeof(struct fimc_is_module_ov5640), GFP_KERNEL);
	if (!module->private_data) {
		err("private_data is NULL");
		ret = -ENOMEM;
		kfree(subdev_module);
		goto p_err;
	}

	ext = &module->ext;
	ext->mipi_lane_num = module->lanes;
	ext->I2CSclk = I2C_L0;
	ext->sensor_con.product_name = SENSOR_NAME_OV5640;
	ext->sensor_con.peri_type = SE_I2C;
	ext->sensor_con.peri_setting.i2c.channel = SENSOR_CONTROL_I2C2;
	ext->sensor_con.peri_setting.i2c.slave_address = 0x6c;
	ext->sensor_con.peri_setting.i2c.speed = 400000;

	ext->from_con.product_name = FROMDRV_NAME_NOTHING;

	ext->companion_con.product_name = COMPANION_NAME_NOTHING;

	module_ov5640 = module->private_data;
	/* get system clock (xclk) frequency */
	ret = of_property_read_u32(dev->of_node, "clock-rates",
				   &module_ov5640->xclk_freq);
	if (!ret) {
		if (module_ov5640->xclk_freq < OV5640_XCLK_MIN ||
		    module_ov5640->xclk_freq > OV5640_XCLK_MAX) {
			dev_err(dev, "invalid xclk frequency: %d\n",
				module_ov5640->xclk_freq);
			return -EINVAL;
		}
	}

	/* get system clock (xclk) */
	module_ov5640->xclk = devm_clk_get(dev, "xclk");
	if (IS_ERR(module_ov5640->xclk)) {
		dev_err(dev, "could not get xclk");
		return -EINVAL;
	}
	clk_set_rate(module_ov5640->xclk, module_ov5640->xclk_freq);
	clk_prepare_enable(module_ov5640->xclk);

	if (client) {
		v4l2_i2c_subdev_init(subdev_module, client, &subdev_ops);
		subdev_module->internal_ops = &internal_ops;
	} else {
		v4l2_subdev_init(subdev_module, &subdev_ops);
	}

	v4l2_set_subdevdata(subdev_module, module);
	v4l2_set_subdev_hostdata(subdev_module, device);
	snprintf(subdev_module->name, V4L2_SUBDEV_NAME_SIZE, "sensor-subdev.%d", module->id);

//sensor_ov5640_init(subdev_module, 1);
p_err:
	info("%s(%d)\n", __func__, ret);
	return ret;

probe_defer:
	if (probe_retried) {
		err("probe has already been retried!!");
		BUG();
	}

	probe_retried = true;
	err("core device is not yet probed");
	return -EPROBE_DEFER;
}

static int sensor_ov5640_remove(struct i2c_client *client)
{
	int ret = 0;
	return ret;
}

#ifdef CONFIG_OF
static const struct of_device_id exynos_fimc_is_sensor_ov5640_match[] = {
	{
		.compatible = "samsung,exynos5-fimc-is-sensor-ov5640",
	},
	{},
};
#endif

static const struct i2c_device_id sensor_ov5640_idt[] = {
	{ SENSOR_NAME, 0 },
};

static struct i2c_driver sensor_ov5640_driver = {
	.driver = {
		.name	= SENSOR_NAME,
		.owner	= THIS_MODULE,
#ifdef CONFIG_OF
		.of_match_table = exynos_fimc_is_sensor_ov5640_match
#endif
	},
	.probe	= sensor_ov5640_probe,
	.remove	= sensor_ov5640_remove,
	.id_table = sensor_ov5640_idt
};

static int __init sensor_ov5640_load(void)
{
        return i2c_add_driver(&sensor_ov5640_driver);
}

static void __exit sensor_ov5640_unload(void)
{
        i2c_del_driver(&sensor_ov5640_driver);
}

module_init(sensor_ov5640_load);
module_exit(sensor_ov5640_unload);

MODULE_AUTHOR("Gilyeon lim");
MODULE_DESCRIPTION("Sensor 8B1 driver");
MODULE_LICENSE("GPL v2");
